#include "face.hpp"
#include "security/key-chain.hpp"
#include "link.hpp"

namespace ndn {
namespace examples {

class Producer : noncopyable
{
public:
  void
  run()
  {
    m_face.setInterestFilter("/",
                             bind(&Producer::onInterest, this, _1, _2),
                             RegisterPrefixSuccessCallback(),
                             bind(&Producer::onRegisterFailed, this, _1, _2));
    m_face.processEvents();
  }

private:
  void
  onInterest(const InterestFilter& filter, const Interest& interest)
  {
    std::cout << "<< I: " << interest << std::endl;

    Link link = interest.getLink();
    std::cout << "\n<< L: " << link << std::endl;

    Link::DelegationSet delegations = link.getDelegations();
    int counter = 0;
    for (auto i = delegations.begin(); i != delegations.end(); i++) {
      counter++;
      if (counter == 1)
      {
        std::cout << "<< Delegation set 1 preference: " << std::get<0>(*i) <<"\n";
        std::cout << "<< Delegation set 1 name: " << std::get<1>(*i) <<"\n";
      }
      if (counter == 2)
      {
        std::cout << "<< Delegation set 2 preference: " << std::get<0>(*i) <<"\n";
        std::cout << "<< Delegation set 2 name: " << std::get<1>(*i) <<"\n\n";
      }
    }

    // Create new name, based on Interest's name
    Name dataName(interest.getName());
    dataName
      .append("testApp") // add "testApp" component to Interest name
      .appendVersion();  // add "version" component (current UNIX timestamp in milliseconds)

    static const std::string content = "HELLO KITTY";

    // Create Data packet
    shared_ptr<Data> data = make_shared<Data>();
    data->setName(dataName);
    data->setFreshnessPeriod(time::seconds(10));
    data->setContent(reinterpret_cast<const uint8_t*>(content.c_str()), content.size());

    // Sign Data packet with default identity
    m_keyChain.sign(*data);
    // m_keyChain.sign(data, <identityName>);
    // m_keyChain.sign(data, <certificate>);

    // Return Data packet to the requester
    std::cout << ">> D: " << *data << std::endl;
    m_face.put(*data);
  }


  void
  onRegisterFailed(const Name& prefix, const std::string& reason)
  {
    std::cerr << "ERROR: Failed to register prefix \""
              << prefix << "\" in local hub's daemon (" << reason << ")"
              << std::endl;
    m_face.shutdown();
  }

private:
  Face m_face;
  KeyChain m_keyChain;
};

} // namespace examples
} // namespace ndn

int
main(int argc, char** argv)
{
  ndn::examples::Producer producer;
  try {
    producer.run();
  }
  catch (const std::exception& e) {
    std::cerr << "ERROR: " << e.what() << std::endl;
  }
  return 0;
}
