#define BOOST_LOG_DYN_LINK 1

#include <iostream>

#include <ndn-cxx/encoding/block-helpers.hpp>
#include <ndn-cxx/util/logger.hpp>
#include <ndn-cxx/util/notification-subscriber.hpp>
#include <ndn-cxx/util/scheduler.hpp>
#include <ndn-cxx/security/command-interest-signer.hpp>
#include <ndn-cxx/security/key-chain.hpp>

#include <ChronoSync/socket.hpp>
//#include <thread>

using namespace ndn;

NDN_LOG_INIT(ChronoApp);

class ChronoApp
{
public:
  ChronoApp(Name pubSubGroupPrefix, Name repoName)
  : m_pubSubGroupPrefix(pubSubGroupPrefix)
  , m_repoName(repoName)
  // /<group-prefix>/repo1/chronosync
  , m_chronoSyncUserPrefix(Name(pubSubGroupPrefix).append(repoName))
  // /<group-prefix>/chronosync - multicast
  , m_chronoSyncPrefix(Name("chronosync"))
  // /localhost/repo1/datastream/insert
  , m_subscriber(m_face, Name("localhost").append("datastream").append(repoName).append("insert"), ndn::time::milliseconds(4000))  // data freshness is 1 second
  {
  }

  ~ChronoApp() {
    m_subscriber.stop();
    m_connection.disconnect();
  }

public:
  void run()
  {
    initialize();

    try {
      m_face.processEvents();
    } catch (std::runtime_error& e) {
      std::cerr << e.what() << std::endl;
      return;
    }
  }

protected:
  void initialize()
  {
    m_chronoSyncSocket =
      std::make_shared<chronosync::Socket>(m_chronoSyncPrefix, m_chronoSyncUserPrefix, m_face,
                                           std::bind(&ChronoApp::onChronoSyncUpdate, this, _1));

    m_connection = m_subscriber.onNotification.connect(std::bind(&ChronoApp::onUpdateFromRepo, this, _1));
    m_subscriber.start();
  }

  void
  onChronoSyncUpdate(const std::vector<chronosync::MissingDataInfo>& v) {
    for (auto ms : v) {
      std::string prefix = ms.session.getPrefix(-1).toUri();

      // Check for our own update like NLSR
      if (prefix == m_chronoSyncUserPrefix.toUri()) {
        continue;
      }

      int seq1 = ms.low;
      int seq2 = ms.high;

      for (int i = seq1; i <= seq2; i++) {
        NDN_LOG_INFO("ChronoSync Update: " << prefix << "/" << i);
        fetchData(ms.session, i, 3);
      }
    }
  }

  void
  fetchData(const Name& sessionName, const uint64_t& seqNo,
            int nRetries)
  {
    Name interestName;
    interestName.append(sessionName).appendNumber(seqNo);

    Interest interest(interestName);
    //interest.setMustBeFresh(true);
    NDN_LOG_INFO("Fetching data for : " << interest.getName() << " " << interest.getNonce());

    m_face.expressInterest(interest,
                           std::bind(&ChronoApp::onData, this, _1, _2),
                           std::bind(&ChronoApp::onNack, this, _1, _2, nRetries),
                           std::bind(&ChronoApp::onTimeout, this, _1, nRetries));
  }

  void
  onData(const Interest& interest, const Data& data)
  {
    Name ds;
    // Content is the name
    ds.wireDecode(data.getContent().blockFromValue());

    // Get seqNumber from the data Name
    uint32_t seq = ds.get(ds.size()-1).toNumber();

    NDN_LOG_INFO("ChronoSync DS Update: " << ds.getPrefix(-1) << "/" << seq);
  }

  void
  onNack(const Interest& interest, const lp::Nack& nack, int nRetries)
  {
    NDN_LOG_INFO("Nack: " << interest.getName() << " " << interest.getNonce());
  }

  void
  onTimeout(const Interest& interest, int nRetries)
  {
    NDN_LOG_INFO("Timeout for interest: " << interest.getName() << " " << interest.getNonce());
    if (nRetries <= 0)
      return;

    Interest newNonceInterest(interest);
    newNonceInterest.refreshNonce();

    m_face.expressInterest(newNonceInterest,
                           std::bind(&ChronoApp::onData, this, _1, _2),
                           std::bind(&ChronoApp::onNack, this, _1, _2, nRetries-1),
                           std::bind(&ChronoApp::onTimeout, this, _1, nRetries-1)
                           );
  }

  void
  onUpdateFromRepo(const Data& data) {
    NDN_LOG_INFO("Got update from repo: " << data.getName());
    // Repo makes sure it streams only its own update
    Name comp = data.getName();
    for (size_t i = 0; i < comp.size(); i++) {
      if (comp.at(i) == m_repoName.at(0)) {
        // update is from our own repo, safe to publish
        NDN_LOG_INFO("ChronoSync Publish: " << data.getName());

        m_chronoSyncSocket->publishData(data.getName().wireEncode(),
                                        time::milliseconds(1000),
                                        m_chronoSyncUserPrefix);
        return;
      }
    }
  }

protected:
  Name m_pubSubGroupPrefix;
  Name m_repoName;

  Name m_chronoSyncUserPrefix;
  Name m_chronoSyncPrefix;

  Face m_face;
  KeyChain m_keyChain;

  std::shared_ptr<chronosync::Socket> m_chronoSyncSocket;

  // Notification from Repo
  util::NotificationSubscriber<Data> m_subscriber;
  util::signal::Connection m_connection;
};

int main(int argc, char* argv[]) {
  if ( argc != 3 ) {
    std::cout << " Usage: " << argv[0]
              << " <pub-sub group prefix> <repoName>\n";
  }
  else {
    Name group(argv[1]);
    ChronoApp ChronoApp(group, Name(argv[2]));
    ChronoApp.run();
  }
}